/**
	Define the language we write cutscenes in. Defines function aliases (I call them short-writes)
	These are not all available cutscene functions, you need to look inside the libraries to discover more.
	These functions are written for Plethora. See plethora.js for all information about the EventQueue.
	See the Wiki (url?) for all (or most) available functions

	This file contains language, scenelets and action definitions.
**/

/*** Generic EventQueue Commands ***/

/** nQ() === Game.EventQueue.add()
 * Add a new event to the Queue, the events will be played back in the same order.
 * @param {blob} event Event to queue. It can be a string, a function with optional parameters, or an action object.
 * @param {this} context If 'event' is a function inside an object, you need to give it 'this', example:
 * EventQueue.add( Game.lithonite.repos, Game.lithonite, 'hero'); // same as running: nQ( "Game.lithonite.repos('hero')" );
 */
nQ = function(event, context) { Game.EventQueue.add.apply(Game.EventQueue,arguments); }

/** uQ() === Game.EventQueue.unshift()
 * Similar to add(), but it will add the event to the beginning of the (running) Queue.
 * more about unshift: It's used when already running the queue, and you are just executing something that
 * at that moment decides that the queue should pause a little for it to finish execution. 
 * multiple unshift activities have to be added in reversed order to be played back correctly.
 */
uQ = function(event, context) { Game.EventQueue.unshift.apply(Game.EventQueue,arguments); }

/** tQ() === Game.EventQueue.addAndDelay()
 * Same as add, but also delays afterwards for delaymsec milliseconds.
 * @param {integer} delaymsec Milliseconds to wait before continuing with the next event
 * @param {blob} event Event to queue. It can be a string, a function with optional parameters, or an action object.
 */
tQ = function(delaymsec, event, context) { Game.EventQueue.addAndDelay.apply(Game.EventQueue,arguments); }

/* fQ() === Game.EventQueue.addAndDelayFrames()
 * Same as add, but also delays afterwards for delayframes frames.
 * param {integer} delayframes Frames to wait before continuing with the next event
 * @param {blob} event Event to queue. It can be a string, a function with optional parameters, or an action object.
 */
fQ = function(delayframes, event, context) { Game.EventQueue.addAndDelayFrames.apply(Game.EventQueue,arguments); }

/* wQ() === Game.EventQueue.wait()
 * Stop executing the EventQueue, until the boolean expression 'boolstr' is true (any non false value, actually).
 * @param {Boolean} boolstr A boolean string
 * @param {Boolean} unshift If we need to add a delay at the beginning, not at the end
 * This actually modifies it's UpdateAndDraw() to a function that runs the 'boolstr' until it is true, then it starts the queue again.
 */
wQ = function(boolstr, unshift) { Game.EventQueue.wait(boolstr, unshift); }

/* dQ() === Game.EventQueue.delayms()
 * Delay the execution of the EventQueue for 'milliseconds' milliseconds.
 * @param {integer} milliseconds Number of milliseconds to delay
 * @param {Boolean} unshift If we need to add a delay at the beginning, not at the end
 */
dQ = function(milliseconds, unshift) { Game.EventQueue.delayms(milliseconds, unshift); }

/* zQ() === Game.EventQueue.delayFrames()
 * Delay the execution of the EventQueue for 'frames' frames.
 * @param {integer} frames Number of frames to delay, range: 1 and up.
 * @param {Boolean} unshift If we need to add a delay at the beginning (of things currently executing), not at the very end
 */
zQ = function(frames, unshift) { Game.EventQueue.delayFrames(frames, unshift); }

/* Wait until a sound has reachead a certain position. The soundfile has to be seekable.
 * @param {integer} snd SoundObject to check its position
 * @param {integer} pos Minimal position on the stream that must be reached to continue
 * @param {Boolean} unshift If we need to add a delay at the beginning (of things currently executing), not at the very end
 */
soundPosWaitQ = function (snd, pos, unshift) { Game.EventQueue.wait(snd+".getPosition()>"+pos, unshift); }

/* Enable/Disable or delay FlipScreen() for Game.EventQueue.autoplay();
 * @param {Boolean} onoff Enable or disable FlipScreen(). To delay x flipscreens, enter a number.
 */
FlipScreenQ = function(onoff) { nQ(Game.EventQueue.SetFlipScreen, Game.EventQueue, onoff); }

/**
 * Play the queued cutscene
 * Start executing the EventQueue. It also continues the queue when paused or waiting.
 * You need to have UpdateAndDraw() defined in a renderscript for it to work. (see plethora.js for more information)
 * @param {string} CutSceneName Name of the cutscene you are playing. Optional. Use 'true' to continue a pause and preserve the cutscene name.
 */
CutScene_Play = function(scene){ Game.EventQueue.play(scene); }

/**
 * Start executing the EventQueue. It also continues the queue when paused or waiting.
 * This version if {@link PlethoraEngine#play} does not depend on a running mapengine on a renderscript or mapengine to work.
 * @param {string} CutSceneName Name of the cutscene you are playing. Optional
 * @param {integer} fps Default frames per second between FlipScreens
 * @param {Boolean} cancancel Set if the cutscene can be aborted by pressing any key.
 * @returns true if cancelled ( EventQueue.cancancel must be true for this to work)
 */
CutScene_AutoPlay = function(CutSceneName, fps, cancancel){ Game.EventQueue.autoplay(CutSceneName, fps, cancancel); }


/*** Lithonite EventQueue Commands ***/

/*
 * Disable keyboard movement, immediately
 */
CutScene_Start=function(){
	Game.lithonite.DetachGIP();
	Game.lithonite.moving = 2;
	SetTalkActivationKey("");
	Game.lithonite.active = false;
}

/*
 * Enable keyboard movement, immediately
 */
CutScene_End = function(){
	Game.lithonite.AttachGIP();
	Game.lithonite.moving = 0;
	SetTalkActivationKey(Game.Keys.talk);
	Game.lithonite.active = true;
}

/*
 * Disable keyboard movement, queued. So the cursor keys dont move the player while the cutscene plays.
 */ 
QCutScene_Start = function() { nQ('CutScene_Start()'); }

/*
 * Enable keyboard movement, queued. So the cursor keys work again.
 */
QCutScene_End = function() { nQ('CutScene_End()'); }


/*** Ainimation EventQueue Commands ***/

/* Queued AddCGMove(). no repeat. Used by Action 'move', use that one (preferred)
 * @param {string} who Name of the person that we want to move/animate
 * @param {string} str Movement string, the commands the sprite will run.
 * @param {Boolean} iPobs If the person ignores Person obstructions (it will wait when obstructed). False by default.
 * @param {Boolean} iTobs If the person ignores tile obstructions (it will wait when obstructed). False by default.
 *
 * The format of str is described throughly in ./script/library/ainimation.js:AddCGMove()
 */
Qmove = function(who, str, iPobs, iTobs) { nQ( Game.lithonite.AddCGMove, Game.lithonite, who, str, (typeof iPobs=='undefined'?true:iPobs), false, (typeof iTobs=='undefined'?true:iTobs)); }


// Queued AddCGMove(). Same as Qmove() but looped.
Qloop = function(who, str, iPobs, iTobs) { nQ( Game.lithonite.AddCGMove, Game.lithonite, who, str, (typeof iPobs=='undefined'?true:iPobs), true, (typeof iTobs=='undefined'?true:iTobs)); }

// Cancel all movement. Note: this function also exists: Game.lithonite.ClearAllPersonCommandsAndCGMoves()
Qclear = function(who, immediate, all) { 
	nQ(Game.lithonite.ClearCGMove, Game.lithonite, who, immediate);
	if(all) nQ(SetPersonScript, null, who, SCRIPT_COMMAND_GENERATOR, "");
}

/** Queued SetPersonDirection
 * @param {string} who Name of the person that we want to change its spriteset image
 * @param {string} dir Direction of the spriteset
 * @param {integer} frame Frame number of the direction. Defaults to 0
 */
Qdirection = function(who, direction, frame){ nQ(Game.lithonite.setDir, Game.lithonite, direction, who, frame); }

/** Plays a full animation sequence into a direction, honouring the frame delays defined in the spriteset. 
 * Does not loop, same as "C" for AddCGMove(), but less demanding for CPU.
 * @param {string} who Name of the person that we want to animate (You can use 0, false or undefined as the value for person if you mean GetCurrentPerson() )
 * @param {string} dir A new direction. Optional.
 * @param {Boolean} last Stay on the last frame of the animation, instead of reverting to frame 0. Defaults to false.
 * note: This does the same as the action 'Animate'.
 */
sQ = function(who, dir, last) { nQ(Game.lithonite.NPCSequence,Game.lithonite,who,dir,last); }

/** 
 * To synchonize movements during a cutscene. We set event markers, single letters with a value.
 * For example: we set a marker to false, and then pause the Game.EventQueue with Zzz() until its true. 
 * EventMarkers will be reset from time to time. Dont store important info in them.
 * Set('X',value) will assign marker 'X' the 'value'.
 * Neg('X') will make marker 'X' false
 * Pos('X') will make marker 'X' true
 * Nul('X') will make marker 'X' 0
 * Zzz('X') will pause the processing of cutscene commands until 'X' is true again
 * Zzz('X',value) will pause the processing of cutscene commands until 'X' == value
 */
Set = function(c,n) { nQ("Game.lithonite.EventMarker['"+c+"']=" + n); }
Neg = function(c) { nQ("Game.lithonite.EventMarker['"+c+"']=false"); }
Pos = function(c) { nQ("Game.lithonite.EventMarker['"+c+"']=true"); }
Nul = function(c) { nQ("Game.lithonite.EventMarker['"+c+"']=0"); }
Zzz = function(c,n) { wQ("Game.lithonite.EventMarker['"+c+"']==" + (n === undefined? true : n) ); }

ClearEventMarker = function() { Game.lithonite.EventMarker = []; }

/*** Teleporthica EventQueue Commands ***/

// Read and set flags. Note: these are not temporal eventmarkers. these values are persistent.
function GetMapFlag(flag, map){ 
	if(!map) map = GetCurrentMap();
	persist.ensureState(map);
	return FLAGS.WORLD[map][flag];
}
function SetMapFlag(flag, value, map){
	if(!map) map = GetCurrentMap();
	persist.ensureState(map);
	return FLAGS.WORLD[map][flag] = value;
}
/**
 * MapChange.active is 0 when not active, 2 when fading out, and 1 just before fading in. (and 3 when seamless-ing)
 */
function WaitMapChange(wait, unshift) { wait=wait||0; wQ( "MapChange.active=="+wait , unshift); }

/*** Rethorica EventQueue Commands ***/


// Text speech short-writes from rethorica.js, will be replaced by KCL
Qsay = function(max, msg, img){Game.EventQueue.add("Game.rethorica.TextBox("+max+",\""+msg+"\","+img+")");}
Qnarrate = function(txt, sec){Game.EventQueue.add("Game.rethorica.printNarration(\""+txt+"\","+sec+")");}
Qsub = function(txt, sec){Game.EventQueue.add("Game.rethorica.printSubtitle(\""+txt+"\","+sec+")");}

//note: Qballoon only can display 1 balloon at the time, for multiple balloons, use StorePersonText() and the "?" commandstring inside a char.
/**
 * Shows a text balloon
 * @param {string} txt
 * @param {integer} sec Seconds to display.
 * @param {string} sprite Name of the target person 
 * @param {integer} pos Position of balloon. Default to -1:balloon is above sprite. 1:below sprite
 */
Qballoon = function(txt,sec,sprite,pos){
	if(txt===false)
		Game.EventQueue.add("Game.rethorica.Balloon.done = function(){return true}");
	else
		Game.EventQueue.add("Game.rethorica.printBalloon(\""+txt+"\","+(typeof sec=='string'?"\""+sec+"\"":sec)+",'"+sprite+"',"+pos+")");
}




// Fill AInimation ACTIONS Object with Actions. The CGMove char to execute an action is: *
// So, to execute a Jump (defined in esthetica.js), we write: "*j". 
// CGMove can do lots of default things, see AInimation.js
Game.lithonite.ACTIONS.j = "Jump()";
Game.lithonite.ACTIONS.J = "Jump('','',true)";
Game.lithonite.ACTIONS.Q = "Quake(30,2)";
Game.lithonite.ACTIONS.h = "hiccup(GetCurrentPerson())";



/* ----- */

/**
 * Instantly move a person to a certain tile, if its not obstructed there (person will be centered on it)
 * example:
 * nQ( {
 *    action: 'ToTile',
 *    sprite: 'hero',
 *    tx: 12,
 *    ty: 4
 * } );
 *
 * @param {number} tx X-axis tile position. Mandatory
 * @param {number} ty Y-axis tile position. Mandatory
 * @param {string} sprite A name of a person (defaults to the current Input Person)
 * @param {Boolean} IgnoreObstructions Ignore obstructions (defaults to true)
 * @param {number} dx Optional X-axis Offset in pixels (defaults to zero)
 * @param {number} dy Optional Y-axis Offset in pixels (defaults to zero)
 * @param {integer} layer Layer to move the person to. Leave empty to not change the layer.
 *
 * The original function can also be queued like this (note: IgnoreObstructions defaults to false):
 * nQ(Game.lithonite.setposTile, Game.lithonite, tx, ty, sprite, IgnoreObstructions, dx, dy, layer);
 */
Game.EventQueue.RegisterAction( 'ToTile', // The 'alias', how we call it.
	Game.lithonite, // The context of the function, the base object in this case: Game.lithonite
	Game.lithonite.setposTile, // The real function we are calling
	['tx', 'ty', 'sprite', 'IgnoreObstructions', 'dx', 'dy', 'layer'], // The order of the parameters
	{ sprite: Game.lithonite.GIP, IgnoreObstructions: true, layer: undefined, dx: 0, dy: 0 }, // The default values for some parameters
	['tx', 'ty'] // These parameters are mandatory
);

/**
 * Instantly move a person to (X,Y) coordinates in pixels
 * @param {number} x X-axis position. Mandatory
 * @param {number} y Y-axis position. Mandatory
 * @param {string} sprite A name of a person, defaults to the current Input Person.
 * @param {Boolean} IgnoreObstructions Ignore obstructions (defaults to true)
 * @param {integer} layer Layer to move the person to. Leave empty to not change the layer.
 */
Game.EventQueue.RegisterAction( 'ToXY',
	Game.lithonite,
	Game.lithonite.setpos,
	['x', 'y', 'sprite', 'IgnoreObstructions', 'layer'],
	{ sprite: Game.lithonite.GIP, IgnoreObstructions: true, layer: undefined },
	['x', 'y'] // These parameters are mandatory
);

/**
 * Offsets a person by a certain number of tiles
 * @param {number} dtx X-axis Offset in tiles, defaults to zero
 * @param {number} dty Y-axis Offset in tiles, defaults to zero
 * @param {string} sprite A name of a person, defaults to the current Input Person.
 * @param {Boolean} IgnoreObstructions Ignore obstructions (defaults to true)
 */
Game.EventQueue.RegisterAction( 'DeltaTile',
	Game.lithonite,
	Game.lithonite.reposTile,
	['dtx', 'dty', 'sprite', 'IgnoreObstructions'], 
	{ sprite: Game.lithonite.GIP, IgnoreObstructions: true, dtx:0, dty:0 }
);

/**
 * Offsets a person by a certain number of pixels
 * @param {number} dx X-axis Offset in pixels, defaults to zero
 * @param {number} dy Y-axis Offset in pixels, defaults to zero
 * @param {string} sprite A name of a person, defaults to the current Input Person.
 * @param {Boolean} IgnoreObstructions Ignore obstructions (defaults to true)
 */
Game.EventQueue.RegisterAction( 'DeltaXY',
	Game.lithonite,
	Game.lithonite.repos,
	['dx', 'dy', 'sprite', 'IgnoreObstructions'],
	{ sprite: Game.lithonite.GIP, IgnoreObstructions: true, dx:0, dy:0 }
);



/**
 * sets a person on the same spot as another, and offset it by a certain amount of pixels. The layer will not change
 * @param {string} other The name of the person where 'sprite' will stand next to. Mandatory.
 * @param {string} sprite A name of a person, defaults to the current Input Person.
 * @param {Boolean} IgnoreObstructions Ignore obstructions (defaults to true)
 * @param {number} dx Optional X-axis Offset in pixels, defaults to zero
 * @param {number} dy Optional Y-axis Offset in pixels, defaults to zero
 */
Game.EventQueue.RegisterAction( 'NextTo',
	Game.lithonite,
	Game.lithonite.reposNextTo,
	['other', 'dx', 'dy', 'sprite', 'IgnoreObstructions'], // Order of parameters given to reposNextTo
	{ sprite: Game.lithonite.GIP, IgnoreObstructions: true, dx: 0, dy: 0 }, // default values
	['other'] // mandatory parameter
);

/**
 * Set the direction of a person
 * @param {string} sprite A name of a person, defaults to the current Input Person.
 * @param {string} direction Direction to set to.
 * This one is identical to SetPersonDirection, but if the direction doesn't exist, it will not Abort Sphere.
 * example:
 * nQ( {
 *    action: 'Direction',
 *    sprite: 'hero',
 *    direction: 'west'
 * } );
 * 
 * You can queue the original SetPersonDirection() Sphere-function like so:
 * nQ(SetPersonDirection, null, 'hero', 'west');
 */
Game.EventQueue.RegisterAction( 'Direction',
	Game.lithonite,
	Game.lithonite.setDir,
	['direction', 'sprite'],
	{ sprite: Game.lithonite.GIP },
	['direction']
);

/**
 * Make a person walk a certain amount of tiles in a certain direction. 
 * @param {string} faceDirection Face in that direction before walking that way (defaults to true)
 * @param {string} direction Direction to move to. Can be 'north', 'northwest', 'east' ... or 'UP', 'UPRIGHT', 'RIGHT' ... (defaults to the current direction, unless the value null is given)
 * @param {integer} tiles Number of tiles to walk in 'direction' (defaults to 1)
 * @param {string} sprite Name of the person that walks (defaults to the input person)
 * @param {Boolean} IgnoreObstructions Should the sprite wait when obstructed? (defaults to true)
 * @param {Boolean} IgnorePersonObstructions Should the sprite wait when obstructed? (defaults to the value of IgnoreObstructions)
 * @param {Boolean} IgnoreTileObstructions Should the sprite wait when obstructed? (defaults to the value of IgnoreObstructions)
 * @param {string/boolean} standstill Should the sprite revert to this direction, frame 0 (standing still). (defaults to false) Set to true for frame 0 of last direction.
 * @param {Boolean} wait Should the EventQueue wait until sprite has finished walking? (defaults to true)
 * @param {2char} startwhen Wait for an event marker to happen. (see Set(), Neg(), Pos() and Nul()). Example: "=X" to wait until 'X' is true, "!X" to wait until 'X' is false.
 * @param {2char} endwith Set an event marker when done. "+X" to set 'X' to true, "-X" to set 'X' to false. Markers must be single ascii characters. 
 *                "<X" To increment 'X' by one, ">X" to decrease 'X' by one.
 * @param {string} mstr Raw movement string we could use to give AddCGMove direct command strings. (see ainimation.js) (executed after the 'direction' commands)
 * Here we use an unnamed function to help us set up AddCGMove()
 *
 * example:
 * nQ( {
 *    action: 'Move',
 *    sprite: 'hero',
 *    direction: 'west',
 *    tiles: 12,
 *    standstill: 'lookwest'
 * } );
 */
Game.EventQueue.RegisterAction( 'Move',
	null,
	function(dir, face, tiles, sprite, obs, iPobs, iTobs, stand, wait, startwhen, endwith, mstr){
		if(iPobs === undefined) iPobs = obs;
		if(iTobs === undefined) iTobs = obs;

		if(startwhen){ 
			Game.lithonite.AddCGMove(sprite, startwhen, iPobs, false, iTobs);
		};

		if(stand === true){
			stand = ",0";
		}else if(stand){
			Game.lithonite.StorePersonDir(sprite, '%', stand);
			stand = ";%";
		}else{
			stand = "";
		};

		if(!dir && (!mstr || tiles))
			dir = GetPersonDirection(sprite);

		if(dir){
			if(dir.length>1){
				dir = dir.replace('DOWN', 'south');
				dir = dir.replace('UP', 'north');
				dir = dir.replace('LEFT', 'west');
				dir = dir.replace('RIGHT', 'east');
				var direction = ['q','n','o','','w','','e','','z','s','m'];
				dir = direction[Game.lithonite.getDirMove(dir) + 5];
			}

			if(dir){
				if(face)
					dir = dir.toUpperCase() + dir + tiles;
				else
					dir = dir + tiles;
			}
		}else
			dir = "";

		Game.lithonite.AddCGMove(sprite, dir + stand + (mstr||"") + (endwith||""), iPobs, false, iTobs);

		if (wait)
			wQ("!GetPersonValue('" + sprite + "','cgmove')" ,true);
	},
	['direction', 'faceDirection', 'tiles', 'sprite', 'IgnoreObstructions', 'IgnorePersonObstructions', 'IgnoreTileObstructions', 'standstill', 'wait', 'startwhen', 'endwith', 'mstr'],
	{ sprite: Game.lithonite.GIP, IgnoreObstructions: true, faceDirection: true, wait:true, standstill:false }
);

/**
 * Make a person emote. Displays a full animation sequence of an emotion.
 * @param {string} sprite Name of the person that will show the emotion
 * @param {string} emotion Available emotions are the directions of spritesets/sfx/emoticons.rss
 * @param {Boolean} wait Should the EventQueue wait until sprite has finished? (defaults to true)
 *
 * example:
 * nQ( {
 *    action: 'Emote',
 *    sprite: 'hero',
 *    emotion: 'exclamation'
 * } );
 */
Game.EventQueue.RegisterAction( 'Emote',
	null,
	function(sprite,emotion,wait){
		if(typeof Game.lithonite.EMOTIONS[emotion] != 'function')
			throw {message:"Unknown emotion '"+emotion+"' for sprite '"+sprite+"'.", filename:'generic.js'};
		var emote = Game.lithonite.EMOTIONS[emotion](sprite);
		if (wait)
			zQ( Game.lithonite.getSeqTimes(emote, emotion) ,true); // Prepend a pause at the beginning of the running eventqueue
	},
	['sprite', 'emotion', 'wait'],
	{sprite: Game.lithonite.GIP, wait:true},
	['emotion']
);

/**
 * Make a person Animate. Displays a full animation sequence starting from the first frame of the given direction.
 * @param {string} sprite Name of the person that will be animated. Mandatory.
 * @param {string} direction Direction in which to animate (default to current direction)
 * @param {Boolean} wait Should the EventQueue wait until sprite has finished? (defaults to true)
 * @param {integer} times How many times to animate a full animation sequence (defaults to 1 time)
 * @param {Boolean} stayonlastframe Do not revert to frame 0 after the animation sequence (defaults to false)
 * @param {Boolean} loop Keep on animating (defaults to false). If set to true, then it will never wait.
 */
Game.EventQueue.RegisterAction( 'Animate',
	null,
	function(sprite, direction, wait, times, loop, last){
		if (loop){
			Game.lithonite.NPCAnimate(sprite, direction);
			return;
		}
		for(var i=0;i<times;i++){
			Game.lithonite.NPCSequence(sprite, direction, last);
		}
		if (wait)
			zQ( times * (Game.lithonite.getSeqTimes(sprite, direction) - (last?1:0)) ,true); // Prepend a pause at the beginning of the running eventqueue
	},
	['sprite', 'direction', 'wait', 'times', 'loop', 'stayonlastframe'],
	{sprite: Game.lithonite.GIP, wait: true, times: 1, loop: false, stayonlastframe: false},
	['sprite']
);


/**
 * Hiccup
 * @param {string} sprite Name of the person to create. required.
 * @param {Boolean} IgnoreObstructions Ignore obstructions while repositioning your sprite (defaults to true)
 */
Game.EventQueue.RegisterAction( 'Hiccup',
	null,
	function(sprite,obs,wait){
		var h = GetPersonValue(sprite, 'height');
		Game.lithonite.AddCGMove(sprite, "*H", obs, false);
		if (wait)
			wQ("GetPersonValue('"+sprite+"','height')=="+h ,true);
	},
	['sprite','IgnoreObstructions','wait'],
	{ sprite: Game.lithonite.GIP, wait: true, IgnoreObstructions:true },
	[]
);

/**
 * To Queue ACTIONS, do this:  nQ(Game.lithonite.ACTIONS['action'], Game.lithonite, sprite);
 */

/**
 * Create a person and place it on the map in a certain direction and on a certain place.
 * @param {string} sprite Name of the person to create. required.
 * @param {string} spriteset Name of the rss file to load. required.
 * @param {Boolean} destroy_with_map If the person should be destroyed with the map. Defaults to true.
 * @param {integer} layer Layer to move the person to. Defaults to the layer of the entrypoint, or the layer of the 'other' person, if used.
 * @param {string} direction Direction to set after creating it. Defaults to the first direction in the spriteset editor (usually north)
 * @param {integer} frame Frame to set. Defaults to 0.
 * @param {string} other Moves our sprite on top of the 'other' sprite. (you probably want to offset it too).
 * @param {number} x X-axis Position in pixels to move sprite to after creation.
 * @param {number} y Y-axis Position in pixels to move sprite to after creation.
 * @param {number} tx X-axis tile position to move sprite to after creation.
 * @param {number} ty Y-axis tile position to move sprite to after creation.
 * @param {number} dx X-axis Offset in pixels, defaults to zero.
 * @param {number} dy Y-axis Offset in pixels, defaults to zero.
 * @param {integer} dtx  X-axis Tile Offset, defaults to zero.
 * @param {integer} dty  Y-axis Tile Offset, defaults to zero.
 * @param {Boolean} IgnoreObstructions Ignore obstructions while repositioning your sprite (defaults to true)
 *
 * example:
 * nQ( {
 *    action: 'CreatePerson',
 *    sprite: 'hero',
 *    spriteset: "hero.rss",
 *    direction: 'west',
 *    tx: 12,
 *    ty: 5
 * } );
 *
 */
Game.EventQueue.RegisterAction( 'CreatePerson',
	null,
	function(sprite, spriteset, destroy_with_map, direction, frame, other, x,y, tx,ty, layer,  dx,dy, dtx,dty, IgnoreObstructions){
		try{
			CreatePerson(sprite, spriteset, destroy_with_map);
			if (direction)
				SetPersonDirection(sprite, direction);
			if (frame)
				SetPersonFrame(person, frame);
			if(x || y)
				Game.lithonite.setpos(
					(x||0) + (dx || 0) + ( dtx===undefined ? 0 : Game.lithonite.TileToMapX(dtx) ),
					(y||0) + (dy || 0) + ( dty===undefined ? 0 : Game.lithonite.TileToMapY(dty) ),
					sprite,
					IgnoreObstructions,
					layer
				);
			else if (other)
				Game.lithonite.reposNextTo(other,
					(dx || 0) + ( dtx===undefined ? 0 : Game.lithonite.TileToMapX(dtx) ),
					(dy || 0) + ( dty===undefined ? 0 : Game.lithonite.TileToMapY(dty) ),
					sprite,
					IgnoreObstructions
				);
			else if (tx || ty)
				Game.lithonite.setposTile(
					(tx || 0) + (dtx || 0),
					(ty || 0) + (dty || 0),
					sprite,
					IgnoreObstructions,
					dx,
					dy,
					layer);
			else if (dx || dy)
				Game.lithonite.repos(
					(dx || 0) + ( dtx===undefined ? 0 : Game.lithonite.TileToMapX(dtx) ),
					(dy || 0) + ( dty===undefined ? 0 : Game.lithonite.TileToMapY(dty) ),
					sprite,
					IgnoreObstructions
				);
			else if (dtx || dty)
				Game.lithonite.reposTile(dtx, dty, sprite, IgnoreObstructions);
		} catch (e) {
			Game.EventQueue.HandleError("CreatePerson Action error: "+this.event+"\n", e);
		};
	},
	['sprite', 'spriteset', 'destroy_with_map', 'direction', 'frame', 'other', 'x', 'y', 'tx', 'ty', 'layer', 'dx', 'dy', 'dtx', 'dty', 'IgnoreObstructions'],
	{IgnoreObstructions: true, destroy_with_map:true},
	['sprite', 'spriteset']
);

/**
 * Test scenelet. Scenelets pause the queue while active.
 */
Game.EventQueue.RegisterScenelet( 'Countdown',
	function(from,to){
		this.from = from;
		this.to = to;
	},
	function(){
		if(this.from == this.to) return true;
		GetSystemFont().drawText(20,10, " SCENELET i="+this.from);
		--this.from;
	},
	function(){},
	['from', 'to'],
	{from: 5, to: 0}
);


// Feek and systemscript screen.js
Game.EventQueue.RegisterScenelet( 'FadeToColor',
        function(msecs, color, bgimage, image, amt ,x1,y1, x2,y2){
                this.time = GetTime();
                this.bgimage = bgimage || GrabImage(0, 0, GetScreenWidth(), GetScreenHeight() );
                this.image = image;
                this.msecs = msecs;
                this.color = CreateColor(color.red, color.green, color.blue, color.alpha);
                this.amt = (amt === undefined) ? 255 : amt;
                this.x = x1 || 0;
                this.y = y1 || 0;
                this.x2 = x2;
                this.y2 = y2;
                this.justdisplay = (this.x2 === undefined && this.y2  === undefined) ? true : false;
        },
        function(){
                this.bgimage.blit(0, 0);

		this.color.alpha = (GetTime() - this.time) * this.amt / this.msecs;
                if (GetTime() - this.time > this.msecs){
                        this.color.alpha = this.amt;
                        if(!this.justdisplay){
                                this.x = this.x2;
                                this.y = this.y2;
                        }
                }

                if(this.image)
                        this.image.blitMask(this.x,this.y, this.color);
                else
                        ApplyColorMask(this.color);

		if (this.color.alpha == this.amt)
			return true;
                if (this.justdisplay)
                        return;

                if (this.x2 > this.x)
                        this.x = (GetTime() - this.time) * this.x2 / this.msecs;
                else if (this.x2 < this.x)
                        this.x = this.x - (GetTime() - this.time) * this.x2 / this.msecs;

                if (this.y2 > this.y)
                        this.y = (GetTime() - this.time) * this.y2 / this.msecs;
                else if (this.y2 < this.y)
                        this.y = this.y - (GetTime() - this.time) * this.y2 / this.msecs;

        },
        function(){},
        ['msecs', 'color', 'bgimage', 'image', 'translucency', 'x1', 'y1', 'x2', 'y2'],
        {msecs: 5, color:CreateColor(0,0,0, 255), translucency:255}
);

Game.EventQueue.RegisterScenelet( 'FadeFromColor',
        function(msecs, color, bgimage, image, amt ,x1,y1, x2,y2){
                this.time = GetTime();
                this.bgimage = bgimage || GrabImage(0, 0, GetScreenWidth(), GetScreenHeight() );
                this.image = image;
                this.msecs = msecs;
                this.color = CreateColor(color.red, color.green, color.blue, color.alpha);
                this.amt = (amt === undefined) ? 255 : amt;
                this.x = x1 || 0;
                this.y = y1 || 0;
                this.x2 = x2;
                this.y2 = y2;
                this.justdisplay = (this.x2 === undefined && this.y2  === undefined) ? true : false;
        },
        function(){
                this.bgimage.blit(0, 0);

                this.color.alpha = 255 - (GetTime() - this.time) * this.amt / this.msecs;
                if (GetTime() - this.time > this.msecs){
                        this.color.alpha = 255 - this.amt;
			if(!this.justdisplay){
				this.x = this.x2;
				this.y = this.y2;
			}
		}

                if(this.image)
                        this.image.blitMask(this.x,this.y, this.color);
                else
                        ApplyColorMask(this.color);

                if (this.color.alpha == 255 - this.amt)
                        return true;
                if (this.justdisplay)
                        return;

                if (this.x2 > this.x)
                        this.x = (GetTime() - this.time) * this.x2 / this.msecs;
                else if (this.x2 < this.x)
                        this.x = this.x - (GetTime() - this.time) * this.x2 / this.msecs;

                if (this.y2 > this.y)
                        this.y = (GetTime() - this.time) * this.y2 / this.msecs;
                else if (this.y2 < this.y)
                        this.y = this.y - (GetTime() - this.time) * this.y2 / this.msecs;

        },
        function(){},
        ['msecs', 'color', 'bgimage', 'image', 'translucency', 'x1', 'y1', 'x2', 'y2'],
        {msecs: 5, color:CreateColor(0,0,0, 255), translucency:255}
);


/**
 * TextBoxes from KCL
 * @param {string} sprite Name of the person that is talking (will use portrait, tint and caption from that person, if defined). 
 * @param {string} portrait Image filename
 * @param {string} caption A name to display below the image
 * @param {string} text The Text to display
 * @param {string} tint A KCL Color (Can also be a Sphere color ;) )
 * @param {string} windowstyle A KCL WindowStyle
 * @param {Object} settings Additional settings for the MessageWindow
 * @param {Boolean} redraw If true, will leave the buffer drawn with the msgbox. Defaults to true
 *
 * If you use 'sprite' then set the values first like so:
 *   SetPersonValue('Daniel', 'caption', "Daniel") 
 *   SetPersonValue('Daniel', 'portrait', "portrait/daniel.png") 
 *   SetPersonValue('Daniel', 'tint', "CreateColor(255,0,255,255)")
 */
Game.EventQueue.RegisterAction( 'Talk',
	null,
	function(sprite, portrait, caption, msg, tint, windowstyle, settings, wait, redraw, pos){
		SetFrameRate(12);
		//var BGimage = new Image(IMAGE_FROM_BUFFER,undefined, 0, 0, GetScreenWidth(), GetScreenHeight()); 
		if(!windowstyle){
			windowstyle = new WindowStyle(Game.gui.windowstyle);
		}
		if(sprite){
			tint = tint || eval(GetPersonValue(sprite, 'tint'));
			portrait = portrait || GetPersonValue(sprite, 'portrait');
			// TODO: Somewhere, get the cached image for the filename string 'portrait'
			caption = caption || GetPersonValue(sprite,'caption');
		}
		if(portrait){
			if(typeof portrait == "string") 
				portrait = new Image(portrait);
			else
				portrait = portrait;
		}else{
			portrait = false;
		}

		if(pos == 'bottom'){
			Game.MSGBOX = new MessageWindow(msg, caption, portrait, false,  false, false, false, 80, new Font(Game.gui.font), windowstyle);
		}else if(pos=='top'){
			Game.MSGBOX = new MessageWindow(msg, caption, portrait, false, 0, 0, false, 80, new Font(Game.gui.font), windowstyle);
		}else{
			Game.MSGBOX = new MessageWindow(msg, caption, portrait, false, 0, pos, false, 80, new Font(Game.gui.font), windowstyle);
		}

		//Game.MSGBOX.background = BGimage;
		if(pos == 'bottom'){
			Game.MSGBOX.portraitWindowAlignment = ALIGNMENT_TOPLEFT;
		}else{
			Game.MSGBOX.portraitWindowAlignment = ALIGNMENT_BOTTOMLEFT;
		}
		Game.MSGBOX.portraitWindow.style = WINDOWSTYLE_NONE;
		Game.MSGBOX.nameWindow.style = WINDOWSTYLE_NONE;

		// Set a new tint color for the text
		if(tint){
			Game.MSGBOX.font.setColorMask(tint);
		}

		// TODO: recursively if objects
		for(var i in settings){
			if(typeof  settings[i] == 'object'){
				for(var j in settings[i]){
					Game.MSGBOX[i][j] = settings[i][j];
				}
			}else
				Game.MSGBOX[i] = settings[i];
		}

		// Display the message box
		Game.MSGBOX.doMessages(msg);

		// Redraw the buffer, so the game doesnt flikker.
		if(IsMapEngineRunning()) RenderMap();
		if(redraw) Game.MSGBOX.redraw();
		SetFrameRate(0);

	},
	['sprite', 'portrait', 'caption', 'text', 'tint', 'windowstyle', 'settings', 'wait', 'redraw','pos'],
	{ settings:{}, wait:true, redraw:true, pos:'bottom' },
	['text']
);

/**
 * Continue with more text, but with the settings from "Talk"
 * @param {string} text The text to display
 * @param {Object} settings Additional settings for the MessageWindow
 * @param {Boolean} redraw If true, will leave the buffer drawn with the msgbox. Defaults to true
 * wait Always true, Eventqueue continues after talking
 */
Game.EventQueue.RegisterAction( 'TalkContinue',
	null,
	function(msg, settings, wait, redraw){
		SetFrameRate(12);

		for(var i in settings){
			if(typeof  settings[i] == 'object'){
				for(var j in settings[i]){
					Game.MSGBOX[i][j] = settings[i][j];
				}
			}else
				Game.MSGBOX[i] = settings[i];
		}
		// Display the message box
		Game.MSGBOX.doMessages(msg);

		// Redraw the buffer, so the game doesnt flikker.
		if(IsMapEngineRunning()) RenderMap();
		if(redraw) Game.MSGBOX.redraw();
		SetFrameRate(0);
	},
	['text', 'settings', 'wait', 'redraw'],
	{ settings:{}, wait:true, redraw:true },
	['text']
);

/**
 * MapChange
 * @param {string} rmp Map filename
 * @param {string} mapname title message to display when map is loaded 
 * @param {string} fade Type of fade (it will only be used for this warp)
 * @param {object} wbinfo Modifies the warp back information. See {@link TeleporthicaEngine#updatePreviousInfo} for details.
 * @param {string} warppoint Warp point you want to teleport to
 * @param {integer} tx Tile X coordinate we want to teleport to. Keep undefined to use the coordinate from the Entry Point.
 * @param {integer} ty Tile Y coordinate we want to teleport to. Keep undefined to use the coordinate from the Entry Point.
 * @param {number} x X coordinate we want to teleport to. use null to use the coordinate we had un the previous map before warping, undefined uses data from entry point
 * @param {number} y Y coordinate we want to teleport to. use null to use the coordinate we had un the previous map before warping, undefined uses data from entry point
 * @param {number} dx X-axis Offset in pixels, defaults to zero.
 * @param {number} dy Y-axis Offset in pixels, defaults to zero.
 * @param {integer} dtx  X-axis Tile Offset, defaults to zero.
 * @param {integer} dty  Y-axis Tile Offset, defaults to zero.
 * @param {integer} layer To the new layer we want to teleport to. Keep undefined to use the layer of the Entry Point.
 * @param {string} run Additional command you want to run, after the map change and the repositioning of the input person and after the run defined in the warppoint
 * @param {Boolean} wait Should the EventQueue wait until Warp has finished? (defaults to true). Set to 1 to wait just before fadein starts.
 * @param {boolean} store Will store the information set with updatePreviousInfo() to a secure place, so we can get it back
 *                  Run this when you just entered a dungeon, Then, when you need to immediately exit the dungeon, just call 
 *                  {@link TeleporthicaEngine#restoreWarpBackInfo} and MapChange.warpBack(true);
 * @param {boolean} restore Will restore the information of MapChange.previous with information stored by {@link TeleporthicaEngine#storeWarpBackInfo}
 * @param {boolean} warpback Set this variable if you want to warp back. Set to false to not run a trigger if we teleport onto one, and true to trigger it.
 *
 * Possible basic teleports are: 0 (disabled fade), black (screen stays black), fade, spot grayfade wipefade stripeout grayfade pixelate and zoom
 */
Game.EventQueue.RegisterAction( 'Warp',
	null,
	function(rmp, mapname, fade, wbinfo, warppoint, tx,ty, x,y, dx,dy, dtx,dty, layer, run, wait, store,restore, warpback){
		if(store){
			MapChange.storeWarpBackInfo();
		}
		if(restore){
			MapChange.restoreWarpBackInfo();
		}
		if(warpback !== undefined){
			return MapChange.warpBack(warpback, mapname, run, fade);
		}
		if(warppoint){
			MapChange.warp(rmp, warppoint, mapname, run, wbinfo, fade);
		}else{
			if(tx !== undefined){
				x = tx<0 ? -1 : MapChange.TileToMapX(tx +(dtx||0));
			}
			if(ty !== undefined){
				y = ty<0 ? -1 : MapChange.TileToMapY(ty +(dty||0));
			}
			if(dx)
				x += dx;
			if(dy)
				y += dy;
			MapChange.warpXY(rmp, x, y, layer, mapname, run, wbinfo, fade);
		}

		if(wait){
			if(wait===true){
				wQ( "!MapChange.active", true);
				
			}else{
				wQ( "MapChange.active=="+wait , true);
			};
		}
	},
	['rmp', 'mapname', 'fade', 'wbinfo', 'warppoint', 'tx','ty', 'x','y','dx','dy','dtx','dty', 'layer', 'run', 'wait', 'store', 'restore', 'warpback'],
	{ wait:true, warpback:undefined},
	['rmp']
);

Game.EventQueue.RegisterAction( 'WarpBack',
	null,
	function(retrigger,msg,run,fade){
		MapChange.warpBack(retrigger,msg,run,fade);
		if(wait){
			if(wait===true)
				wQ( "!MapChange.active", true);
			else
				wQ( "MapChange.active=="+wait , true);
		}
	},
	['retrigger', 'mapname', 'run', 'fade', 'wait'],
	{ retrigger:false, wait:true },
	[]
);


/**
 * Spiral around and maybe towards and get sucked into a spot. While possibly spinning, rotating and screaming.
 * @param {string} sprite Name of the person to create. required.
 * @param {number} x X-axis Centerpoint Position in pixels to move sprite to
 * @param {number} y Y-axis Centerpoint Position in pixels to move sprite to
 * @param {string} other Moves our sprite around the 'other' sprite. (replaces x and y parameters)
 * @param {float} anglespeed Speed in radians for rotating around the centerPoint
 * @param {float} radialspeed Speed of the radius to the centerpoint, a negative number to spiral towards the centerpoint
 * @param {float} shrinkspeed Speed at which to make Daniel smaller
 * @param {float} rotatespeed Speed at wicht Daniel rotates around himhelf
 * @param {Boolean} IgnoreObstructions Ignore obstructions while repositioning your sprite (defaults to true)
 * @param {Boolean} restore Restores angle and ScaleFactor to 0 and 1 respectively
 * @param {funtion} stop This function returns true when we have to stop this scenelet (Defaults to a radius smaller than 2)
 */
Game.EventQueue.RegisterScenelet( 'SpiralTowards',
	function(x,y,other,sprite,anglespeed,radialspeed,shrinkspeed,rotatespeed,IgnoreObstructions, restore,stop){
		if(other){
			x = GetPersonX(other);
			y = GetPersonY(other);
		}
		this.x0 = x;
		this.y0 = y;
		this.x = GetPersonX(sprite);
		this.y = GetPersonY(sprite);
		this.sprite = sprite;
		this.IgnoreObstructions = IgnoreObstructions;
		this.dangle = anglespeed;
		this.dradius= radialspeed;
		this.rotate = rotatespeed;
		this.shrink = shrinkspeed;
		this.ScaleFactor = 1;
		this.restore= restore;

		this.heighthalf = GetPersonValue(this.sprite,'height')>>1;
		this.widthhalf = GetPersonValue(this.sprite,'width')>>1;

		var H = this.x0 - this.x;
		var V = this.y0 - this.y;
		this.radius = Math.sqrt(H*H + V*V);
		this.angle = Math.atan2(V, H);
		this.stop = stop || function(){if(this.radius<2)return true;return false};
	},
	function(){
		SetPersonXYFloat(this.sprite, this.x, this.y);
		if(this.rotate){
			SetPersonAngle(this.sprite, this.rotate + GetPersonAngle(this.sprite));
		};
		if(this.shrink){
			// We dont have GetPersonScaleFactorX and GetPersonScaleFactorY :(
			this.ScaleFactor += this.shrink;
			SetPersonScaleFactor(this.sprite, this.ScaleFactor, this.ScaleFactor);
		};
		this.angle += this.dangle;
		this.radius += this.dradius;

		this.x = this.x0 + this.radius * Math.cos(this.angle); // r*cos(theta). angle in degrees, theta in radians
		this.y = this.y0 - this.radius * Math.sin(this.angle);
		if(this.shrink){
			this.y += this.heighthalf * (1 - this.ScaleFactor);
			this.x += this.widthhalf * (1 - this.ScaleFactor);
		}
		return this.stop();
	},
	function(){
		if(this.restore){
			SetPersonScaleFactor(this.sprite, 1,1);
			SetPersonAngle(this.sprite, 0);
		}
	},
['x', 'y', 'other', 'sprite', 'anglespeed', 'radialspeed', 'shrinkspeed', 'rotatespeed', 'IgnoreObstructions','restore','stop'],
{ sprite: Game.lithonite.GIP, anglespeed: 0.05, radialspeed:-0.1, shrinkspeed:0, rotatespeed:0,  IgnoreObstructions: true, restore:true },
[]
);
